function [W_eq_fd, W_eq] = nylon_W_eq_fd(Temps, Pw, range, changepoints)
%
% Returns W_eq as a functional data object
%
% This function combines equations 2 and 3 in Zheng et al,2005 Ind. End.
%   Chem. Res., Vol 44 No 8
% Pw is the partial pressure of water in the vapor phase in equilibrium
%   with the molten polymer.  It is assumed to be a step function.
% Temps is the Tempserature of the system, a constant
% W_EQ is the water content of the nylon melt at both chemical and
%   vapor-liquid equilibrium.  
% obs_time is a vector of the same length as Pw
% CHANGEPOINTS are assumed to be unique from the observation times 
% The output is a step function fdobject
% 
%
if(isa(Temps,'cell'))
   Temps = cell2mat(Temps);
end
if(~isa(range,'cell'))
    obs = cell(1,1);
    obs{1} = range;
    range = repmat(obs,length(Temps),1);
end

if(~isa(changepoints,'cell'))
    ch = cell(length(Temps),1);
    for i=1:length(Temps)
        ch{i}=changepoints(i,:);
    end
    changepoints=ch;
end    
if(~isa(Pw,'cell'))
    P_Temps = cell(length(Temps),1);
    if(size(Pw,2)==length(Temps))% if true then then cols of Pw represent runs 
        Pw=Pw';
    end
    for i=1:length(Temps)
        P_Temps{i}=Pw(i,:);
    end
    Pw=P_Temps;
end

% constants = [T_0, k_m,gamma_w0,Tc,Pc,R]
%               1    2    3       4  5 6
constants    = [549, 24.3, 20.97, 647.3, 165376,8.3145e-3];

Tc = constants(4);
Pc = constants(5);
nquad=5;% quadrature value points
norder = 1; %spline so it is piecewise continuous

% equation 3, the Wagner equation
W_eq_fd = cell(length(Temps),1);
W_eq = cell(length(Temps),1);
for i=1:length(Temps)
    % equation 3, the Wagner equation
    P_w_sat = Pc* exp(  ( -7.77224*(1-Temps(i)/Tc)   +1.45684*(1-Temps(i)/Tc).^1.5 -...
                    2.71492*(1-Temps(i)/Tc).^3 -1.41336*(1-Temps(i)/Tc).^6 )./(Temps(i)/Tc) );
    % equation 2
    W_eq{i} = (5.55e4)*(Pw{i}./P_w_sat).*exp(-9.624+3613./Temps(i));
    quadvals = MakeQuadPoints(sort([range{i},changepoints{i}]),nquad);
    knots = sort([range{i},changepoints{i}]);
    nbasis   = length(knots) + norder - 2;
    basisobj = create_bspline_basis( range{i},nbasis, norder, knots,[], quadvals);
    Lfd = fdPar(basisobj,0);
    [sorted, index]=sort(W_eq{i});
    uniqueW_eq=unique(W_eq{i});
    pointw=[];
    for j=1:length(unique(W_eq{i}))
        pointw=[pointw,min(index(find(uniqueW_eq(j)==sorted)))   ];
    end
    pointw=sort(pointw);
    W_eq{i}=reshape(W_eq{i},length(W_eq{i}),1);
    if(W_eq{i}(1)==W_eq{i}(end))
        W_eq_fd{i} = smooth_basis(sort([range{i},changepoints{i}]),[W_eq{i}(pointw);W_eq{i}(end);W_eq{i}(end)],Lfd);
    else
            W_eq_fd{i} = smooth_basis(sort([range{i},changepoints{i}]),[W_eq{i}(pointw);W_eq{i}(end)],Lfd);
    end
end

