%% Demonstration of Profiled Estimation of the Nylon Differential Equations
%
%
% This page demonstrates profiling estimation code for differential
% equations with a positive constraint on the smooth.  We compare the
% results from a positive smoothing constraint with an unconstrained system.
%
% The nylon data is from  Zheng et al 2005 using a 4 parameter version of
% their differential equations and rescaled parameters as described in Campbell et
% al. 2007.  These equations include calculations of $W_{eq}$  and $C_T$ from the input
% water pressure using equations from Schaffer et al. 2003.  
%
% David Campbell, Giles Hooker,...............
%
% Wei Zheng, Kim B. McAuley, E. Keith Marchildon and
% Yao K. Zhen, 2005, "Effects of End-Group Balance on Melt-Phase Nylon 612 
% Polycondensation: Experimental Study and Mathematical Model", Ind. Eng. Chem. Res.
%  vol 44, p 2675 - 2686
%
% M.A. Schaffer, Kim B. McAuley, M.F. Cunningham, and E. Keith Marchildon, 
% 2003 "Experimental Study and Modeling of Nylon Polycondensation in the
% Melt Phase", Ind. Eng. Chem. Res. 42, p2946  
%
%The nylon differential equation system:
%
%$$ \frac{dA}{dt} = -k_p*10^{-3}*(AC-LW/K_a)$$
%$$ \frac{dC}{dt} = -k_p*10^{-3}*(AC-LW/K_a)$$
%$$ \frac{dW}{dt} = k_p*10^{-3}*(AC-LW/K_a) - k_m(W-W_{eq})$$
%
% $k_m$ is a known constant from Schaffer et al. 2003, $K_p$ is unknown and
% $K_a$ is a function of the three other  parameters $[\gamma, K_{a0},
% \Delta H]$, experimentally controlled temperature $T$, reference
% temperature $T_0=549.15$, $R=8.3145*10^{-3}$ and
% $C_{T}=20.97\exp[-9.624+3613/T]$ 
%
% $$ K_a = (1+W_{eq}\gamma 10^{-3})C_T\color{blue}K_{a0}\color{black} \exp\Big[-\frac{\color{blue}\Delta H\color{black}}{R}\Big(1/T-1/T_0\Big)\Big] $$
%
%
% We use the surrix '_pos' to denote the inputs and results specific to
% using a positive constraint on the smooth and '_unc' to denote those from
% an unconstrained smooth. Note that for the unconstrained case where all 
% equations are differential and not algebraic we can pass an empty vector
% instead of equation_info to specify our conditions.
%

clear
odefn = @nylonfn1;         % Function for ODE solver (exact), used to simulate data
                         
fn.fn = @nylonfn;              % RHS function
fn.dfdx = @nylondfdy;          % Derivative wrt inputs (Jacobian)
fn.dfdp = @nylondfdpar;        % Derviative wrt parameters
fn.d2fdx2 = @nylond2fdy2;      % Hessian wrt inputs
fn.d2fdxdp = @nylond2fdydpar;  % Hessian wrt inputs and parameters
fn.d2fdp2 = @nylond2fdpar2;    % Hessian wrt parameters.      
fn.d3fdx3 = @nylond3fdy3;      % Third derivative wrt inputs.
fn.d3fdx2dp = @nylond3fdy2dp;  % Third derivative wrt intputs, inputs and pars.
fn.d3fdxdp2 = @nylond3fdydp2;  % Third derivative wrt inputs, pars and pars. 


equation_info.alg = [1,1,1];        % None of the 3 DEs are algebrais equations
equation_info.pos = [1,1,1];        % All positive constraints
%% Set up the parameters and starting values.
%
%parameters = [ K_p0*1000  E  g  beta*1e-3 Ka0   deltaH]
%
% parameters which are functions of $k_p$ and their units
%
% k_p0    Mg $mol^{-1} h^{-1}$
% E       kJ $mol^{-1}$
%
% parameters which are functions of $K_a$ and their units
% g        L/mol
% beta     degrees kelvin
% Ka0      no units
% deltaH   kJ $mol^{-1}$                
%

state=15;  % set the random seed to reproduce these pseudo random results
rand('state',state)
randn('state',state)

truepars  = [15, 10]';
startpars = (normrnd(truepars,3));


%% Set up Experimental Conditions
%
% The reactor is started with one value of $P_w$, at the first changepoints the
% value of $P_w$ decreases and returns ot the initial value after the
% second changepoint.  A step function is used to change $P_w$.
%
% All experimental runs occur at a constant temperature $Temps$
%
% While this example is set up for a single run, you can try multiple runs
% by changing Nruns to a larger integer.
%

Nruns = 1;                          % change this to try a multiple run system
changepoints=cell(Nruns,1);
Temps=ones(Nruns,1);
Pw=cell(Nruns,1);
for i=1:Nruns
    changepoints{i}=repmat(normrnd([3,7],.25),Nruns,1);
    Temps(i) = floor(normrnd(544,8)); 
    Pw{i}    = [760,normrnd(75,10),760];
end


%% Set observation times and the experimental settings.
% We set up the observation times allowing A to be observed more often than
% C. W was not observed at all.  We set up Tcell to have all the possible
% observation times, then produce a logical index to decide the times at
% which C was observed.
%
% Each run is performed at a constant temperature with input $P_w$
% manipulated via a step change occuring at two changepoints.  $P_w$ is
% initially at a high level, then steps doen and steps back to its original
% value.  $P_w$ has a one to one correspondence to $W_eq$, and this gets
% passed as a functional data object along with temperature and the
% index for the positive constraint to the DE functions.
%

sigma=.05;
P_observe_C = .85;
% times of observations
Tcell = cell(Nruns,3);
C_time_index = cell(Nruns,3);
tfine = cell(Nruns,1);
for i=1:Nruns
    Tcell{i,1} = linspace(0,10,poissrnd(30));
    C_time_index{i}  = rand(length(Tcell{i,1}),1)<P_observe_C;
    tfine{i} = linspace(0,10,200);
end
tfine = repmat(tfine,1,3);

time_temp=cell(Nruns,1);
range=cell(Nruns,1);
for i = 1:Nruns
    time_temp{i}=[Tcell{i,1}(1),changepoints{i},Tcell{i,1}(end)];
    range{i} = [Tcell{i,1}(1),Tcell{i,1}(end)];
end

[W_eq W_eqtrue]=  nylon_W_eq_fd(Temps, Pw,range, changepoints);
W_eq_temp = eval_fdcell(time_temp{1},W_eq,0);

fn_extras_pos=cell(Nruns,3);
fn_extras_unc=cell(Nruns,3);
for i = 1:Nruns
    fn_extras_pos(i,:)=[{Temps(i)},{W_eq{i}},{equation_info.pos}];
    fn_extras_unc(i,:)=[{Temps(i)},{W_eq{i}},{[0,0,0]}];
end


%% Solve the nylon equations to simulate some data
%
% The ODE must be solved in 3 pieces, one for each value of $P_w$.  The
% solution pieces are constrained to form a single continuous solution.
% Random normal noise is added to the solution to make observations.
%

%ode solver options
odeopts = optimset('DerivativeCheck','off','Jacobian','on',...
    'Display','iter','MaxIter',1000,'TolFun',1e-8,'TolX',1e-10);

% initial values for the solver:
y0 = [poissrnd(50,Nruns,1),poissrnd(125,Nruns,1),W_eq_temp{i}(1)];

plot_Ycell = cell(Nruns,1);
plot_Ycell1 = cell(Nruns,1);
plot_Ycell2 = cell(Nruns,1);
plot_Ycell3 = cell(Nruns,1);
plot_time = cell(Nruns,1);
plot_time1 = cell(Nruns,1);
plot_time2 = cell(Nruns,1);
plot_time3 = cell(Nruns,1);

for i=1:Nruns
    [plot_time_temp,plot_Ycell_temp] =   ode45(odefn,0:8,y0(i,:),odeopts,truepars,[fn_extras_pos{i},W_eq_temp{i}(1)]');
% Although we can start from any initial values, if we use a set of initial
% values which are close to the steady state, our simulated data will look
% more like the data in the referenced paper.
    [plot_time1{i},plot_Ycell1{i}] = ode45(odefn,Tcell{i,1}(Tcell{i,1}<=changepoints{i}(1)),plot_Ycell_temp(end,:)-[5,5,0],odeopts,truepars,[fn_extras_pos{i,1},W_eq_temp{i}(1)]');
    [plot_time2{i},plot_Ycell2{i}] = ode45(odefn,Tcell{i,1}(Tcell{i,1}>=plot_time1{i}(end)&Tcell{i,1}<=changepoints{i}(2)),plot_Ycell1{i}(end,:),odeopts,truepars,[fn_extras_pos{i,1},W_eq_temp{i}(2)]');
    [plot_time3{i},plot_Ycell3{i}] = ode45(odefn,Tcell{i,1}(Tcell{i,1}>=plot_time2{i}(end)),plot_Ycell2{i}(end,:),odeopts,truepars,[fn_extras_pos{i,1},W_eq_temp{i}(3)]');

    plot_time{i}    = [plot_time1{i}',plot_time2{i}(2:end)',plot_time3{i}(2:end)']';
    plot_Ycell{i}   = [plot_Ycell1{i}',plot_Ycell2{i}(2:end,:)',plot_Ycell3{i}(2:end,:)']';
    
    Ycell{i,1} = normrnd(plot_Ycell{i}(:,1),sigma);
    Ycell{i,2} = normrnd(plot_Ycell{i}(C_time_index{i},2),sigma);
    Ycell{i,3} = plot_Ycell{i}(:,3);
    Ycell{i,3} = [];
    Tcell{i,2} = Tcell{i,1}(C_time_index{i});
    Tcell{i,3} = [];
end

%% Plot observations and order the plots by temperature


[ii,sort_index]=sort(Temps);
for k = 1:Nruns
    i=sort_index(k);
    axes1 = axes(...
        'FontSize',16,'Position',[.12,.1,.84,.84],...
        'XMinorTick','on','Parent',figure(2));
%     h=plot(Tcell{i,1},Ycell{i,1},'r',Tcell{i,2},Ycell{i,2},'b',Tcell{i,1},Ycell{i,1},'.r',Tcell{i,2},Ycell{i,2},'.b','MarkerSize',20,'LineWidth',2);
    subplot(Nruns,3,(i-1)*3+1)
    plot(Tcell{i,1},Ycell{i,1},'.',plot_time{i},plot_Ycell{i}(:,1),'g')
    ylabel(strcat('T=',num2str(Temps(i))),'FontSize',20)
    subplot(Nruns,3,(i-1)*3+2)
    plot(Tcell{i,2},Ycell{i,2},'.',plot_time{i},plot_Ycell{i}(:,2),'g')
    subplot(Nruns,3,(i-1)*3+3)
    plot(Tcell{i,3},Ycell{i,3},'.',plot_time{i},plot_Ycell{i}(:,3),'g')

    xlim([0,Tcell{i}(end)])
end


%% Profiling optimisation control

maxit1 = 200;      % Maximum iterations interior of profiling
maxit0 = 500;     % Maximum iterations for outer optimization


lsopts_out = optimset('DerivativeCheck','off','Jacobian','on',...
    'Display','iter','MaxIter',500,'TolFun',1e-14,'TolX',1e-14);

% Optimiation control within profiling
lsopts_in = optimset('DerivativeCheck','off','Jacobian','on',...
    'Display','off','MaxIter',500,'TolFun',1e-14,'TolX',1e-14,...
    'JacobMult',@SparseJMfun);

nquad = 5;       % No. between-knots quadrature points.

%% Setting up functional data objects and perform a model based smooth.
%
% Set up the B-spline knots, quadrature points and bases.  We use enough knots
% to allow a break in the first derivative at the step changes in $P_w$.
%
% We also get an initial estimate of the coefficients using a penalty on
% the second derivatives.
%
% In the referenced paper the approximate measurement error is known so we
% include it in the term $wts$.
%
% Note that when we plot the data, the smooth often behaves poorly in a
% small neighbourhood of the break in the derivative.  Consequently we omit
% an open $\delta$ neighbouhood from the integral in the spline fitting
% criteria.  The boundaries of this neighbourhood are shown in the plot of
% the smooth.

norder = 4;                 % Order of B-spline approximation 
knots_cell  = cell(size(Tcell));
for i=1:Nruns
    knots_cell(i,:) = repmat({linspace(0,10,2*length(Tcell{i,1}))},Nruns,3);
    for j=1:(size(Tcell,2))
        break_knots=sum(knots_cell{i,j}==changepoints{i}(1));
        if(norder-1-break_knots>0)
            knots_cell{i,j}=sort([knots_cell{i,j},repmat(changepoints{i},1,(norder-1-break_knots))]);
        end
        if(knots_cell{i,j}(end)~=range{i}(2))
            knots_cell{i,j}=[knots_cell{i,j};range{i}(2)];
        end
    end
end

% Set up bases
LFD_pen_order=2;
lambda = 1e2;
lambda_first = 1e-4;
rough_ord = 2; 
delta = 1e-1; % half the open neighbourhood around the changepoints that we omit from the integrated penalty.
wts=repmat([1/0.6^2,1/2.4^2,0],Nruns,1); % this is 1/sigma_A, 1/sigma_C and 0 for [W] since it's unobserved

for i=1:Nruns

[DEfd_cell_pos(i,:),resnorm_pos(i)] = SplineEst(fn,Tcell(i,:),Ycell(i,:),startpars,...
    knots_cell(i,:),wts,lambda,lambda_first,rough_ord,equation_info,lsopts_in,...
    [],fn_extras_pos,norder,changepoints{i},delta);
[DEfd_cell_unc(i,:),resnorm_unc(i)] = SplineEst(fn,Tcell(i,:),Ycell(i,:),startpars,...
    knots_cell(i,:),wts,lambda,lambda_first,rough_ord,[],lsopts_in,...
    [],fn_extras_unc,norder,changepoints{i},delta);

end

% Plot results along with data and show the region that is omitted from the
% integral of the spline criterion.

devals_pos = eval_fdcell(tfine,DEfd_cell_pos,0,equation_info.pos);
devals_unc = eval_fdcell(tfine,DEfd_cell_unc,0);
figure
[Ts indys]=sort(Temps);
for k = 1:Nruns
    i=indys(k);
    for j = 1:3
        subplot(size(Ycell,1),size(Ycell,2),(size(tfine,1)-k)*size(Ycell,2)+j)
        if(~isempty(Tcell{i,j}))
            plot(tfine{i},devals_pos{i,j},'g',tfine{i},devals_unc{i,j},'b',Tcell{i,j},Ycell{i,j},'k.','LineWidth',2);
            hold on
            plot([changepoints{i}(1)-delta,changepoints{i}(1)-delta],[min(Ycell{i,j})-10,max(Ycell{i,j})+10],'--r',...
                 [changepoints{i}(1)+delta,changepoints{i}(1)+delta],[min(Ycell{i,j})-10,max(Ycell{i,j})+10],'--r',...
                 [changepoints{i}(2)-delta,changepoints{i}(2)-delta],[min(Ycell{i,j})-10,max(Ycell{i,j})+10],'--r',...
                 [changepoints{i}(2)+delta,changepoints{i}(2)+delta],[min(Ycell{i,j})-10,max(Ycell{i,j})+10],'--r','LineWidth',1);
            hold off
        else
        plot(tfine{i},devals_pos{i,j},'g',tfine{i},devals_unc{i,j},'b','LineWidth',2);
        hold on
        plot([changepoints{i}(1)-delta,changepoints{i}(1)-delta],[min([devals_pos{i,j};devals_unc{i,j}])-10,max([devals_pos{i,j};devals_unc{i,j}])+10],'--r',...
             [changepoints{i}(1)+delta,changepoints{i}(1)+delta],[min([devals_pos{i,j};devals_unc{i,j}])-10,max([devals_pos{i,j};devals_unc{i,j}])+10],'--r',...
             [changepoints{i}(2)-delta,changepoints{i}(2)-delta],[min([devals_pos{i,j};devals_unc{i,j}])-10,max([devals_pos{i,j};devals_unc{i,j}])+10],'--r',...
             [changepoints{i}(2)+delta,changepoints{i}(2)+delta],[min([devals_pos{i,j};devals_unc{i,j}])-10,max([devals_pos{i,j};devals_unc{i,j}])+10],'--r','LineWidth',1);
        hold off
        ylim([floor(min([devals_pos{i,j};devals_unc{i,j}])/5)*5,ceil(max([devals_pos{i,j};devals_unc{i,j}])/5)*5])
        end
        if(j==1)
            ylabel(num2str(Temps(i)),'FontSize',20)
            Yrange = [floor(min([devals_pos{i,j};devals_unc{i,j};Ycell{i,1}])/5)*5,ceil(max([devals_pos{i,j};devals_unc{i,j};Ycell{i,1}])/5)*5];
            ylim(Yrange);
        elseif(j==2)
            Ytemp = diff(Yrange)/2;
            Ymid = mean([min([devals_pos{i,j};devals_unc{i,j};Ycell{i,j}]),max([devals_pos{i,j};devals_unc{i,j};Ycell{i,j}])]);
            ylim([Ymid-Ytemp,Ymid+Ytemp])         % Give A and C the same vertical scaling, since C acts like it is just a vertical shift of A.
        end
        xlim([0,Tcell{i,1}(end)])
        if(i==1 && j==2)
            title('Model based smooth, pos (-g), unc (-b), data (.k), \delta neighbourhood (--r)')
        end
    end
end


%% Perform the Profiled Estimation 


tic
[newpars_pos,newDEfd_cell_pos] = Profile_GausNewt(startpars,lsopts_out,...
    DEfd_cell_pos,fn,lambda,Ycell,Tcell,wts,equation_info,lsopts_in,fn_extras_pos);
toc
tic
[newpars_unc,newDEfd_cell_unc] = Profile_GausNewt(startpars,lsopts_out,...
    DEfd_cell_unc,fn,lambda,Ycell,Tcell,wts,[],lsopts_in,fn_extras_unc);
 toc
disp('Initial values, Positively constrained, Unconstrained, Truth')
disp(num2str([startpars';newpars_pos';newpars_unc';truepars']))

%% plot the results
 
devals_pos = eval_fdcell(tfine,newDEfd_cell_pos,0,1);
    devals = eval_fdcell(tfine,newDEfd_cell_unc,0);
figure
[Ts indys]=sort(Temps);
for k = 1:size(Temps)
    i=indys(k);
    for j = 1:size(Ycell,2)
        subplot(size(Ycell,1),size(Ycell,2),(k-1)*size(Ycell,2)+j)
        plot(tfine{i},devals_pos{i,j},'g',tfine{i},devals{i,j},'b',Tcell{i,j},Ycell{i,j},'k.','LineWidth',2);
        hold on
        plot([changepoints{i}(1)-delta,changepoints{i}(1)-delta],[min(devals_pos{i,j})-10,max(devals_pos{i,j})+10],'--r',...
              [changepoints{i}(1)+delta,changepoints{i}(1)+delta],[min(devals_pos{i,j})-10,max(devals_pos{i,j})+10],'--r',...
              [changepoints{i}(2)-delta,changepoints{i}(2)-delta],[min(devals_pos{i,j})-10,max(devals_pos{i,j})+10],'--r',...
              [changepoints{i}(2)+delta,changepoints{i}(2)+delta],[min(devals_pos{i,j})-10,max(devals_pos{i,j})+10],'--r','LineWidth',1);
        hold off
        ylim([floor(min([devals_pos{i,j};devals_unc{i,j}])/5)*5,ceil(max([devals_pos{i,j};devals_unc{i,j}])/5)*5])
        if(j==1)
            ylabel(num2str(Temps(i)),'FontSize',20)
        end
        xlim([0,max([Tcell{i,1}(end);Tcell{i,2}(end)])])
    if(k==1 && j==2)
        title('Model based smooth, pos (-g), unc (-b), data (.k), \delta neighbourhood (--r)')
    end
    end
end

figure
[Ts indys]=sort(Temps);
for k = 1:size(Ycell,1)
    i=indys(k);
    for j = 1:size(Ycell,2)
        subplot(size(Ycell,1),size(Ycell,2),(k-1)*size(Ycell,2)+j)
        plot(tfine{i},(devals_pos{i,j}-devals{i,j}),'b','LineWidth',2);
        hold on
        plot([changepoints{i}(1)-delta,changepoints{i}(1)-delta],[min(devals_pos{i,j}-devals{i,j})-10,max(devals_pos{i,j}-devals{i,j})+10],'--r',...
              [changepoints{i}(1)+delta,changepoints{i}(1)+delta],[min(devals_pos{i,j}-devals{i,j})-10,max(devals_pos{i,j}-devals{i,j})+10],'--r',...
              [changepoints{i}(2)-delta,changepoints{i}(2)-delta],[min(devals_pos{i,j}-devals{i,j})-10,max(devals_pos{i,j}-devals{i,j})+10],'--r',...
              [changepoints{i}(2)+delta,changepoints{i}(2)+delta],[min(devals_pos{i,j}-devals{i,j})-10,max(devals_pos{i,j}-devals{i,j})+10],'--r','LineWidth',1);
        hold off
        ylim([floor(min([devals_pos{i,j}-devals{i,j}])/2)*2,ceil(max([devals_pos{i,j}-devals{i,j}])/2)*2])
        if(j==1)
            ylabel(num2str(Temps(i)),'FontSize',20)
        end
        xlim([0,max([Tcell{i,1}(end);Tcell{i,2}(end)])])
        if(i==1 && j==2)
            title('Positive constraint results - Unconstrained results, \delta neighbourhood (--r)')
        end
    end
end

%% Calculate a Sample Information Matrix


% Hessian of squared error with respect to parameters

d2Jdp2_pos = make_d2jdp2(newDEfd_cell_pos,fn,Ycell,Tcell,lambda,newpars_pos,equation_info,wts,fn_extras_pos);
d2Jdp2_unc = make_d2jdp2(newDEfd_cell_unc,fn,Ycell,Tcell,lambda,newpars_unc,[],wts,fn_extras_unc);

% Second derivatives with respect to parameters and observations

d2JdpdY_pos = make_d2jdpdy(newDEfd_cell_pos,fn,Ycell,Tcell,lambda,newpars_pos,equation_info,wts,fn_extras_pos);
d2JdpdY_unc = make_d2jdpdy(newDEfd_cell_unc,fn,Ycell,Tcell,lambda,newpars_unc,[],wts,fn_extras_unc);

% Resulting derivative of parameters with respect to observations

dpdY_pos = -d2Jdp2_pos\d2JdpdY_pos;
dpdY_unc = -d2Jdp2_unc\d2JdpdY_unc;

% Variance of observations:

S_pos = make_sigma(newDEfd_cell_pos,Tcell,Ycell,0,equation_info.pos);
S_unc = make_sigma(newDEfd_cell_unc,Tcell,Ycell,0);

% Resulting parameter covariance matrix:

Cov_pos = dpdY_pos*S_pos*dpdY_pos';
Cov_unc = dpdY_unc*S_unc*dpdY_unc';

 
upper_pos=newpars_pos+2*sqrt(diag(Cov_pos));
upper_unc=newpars_unc+2*sqrt(diag(Cov_unc));
lower_pos=newpars_pos-2*sqrt(diag(Cov_pos));
lower_unc=newpars_unc-2*sqrt(diag(Cov_unc));

CI_pos=[upper_pos,newpars_pos,lower_pos];
CI_unc=[upper_unc,newpars_unc,lower_unc];

disp('Approximate 95% Confidence Interval for Parameters under Positive Constraint')
disp(num2str(CI_pos));
disp('Approximate 95% Confidence Interval for Parameters Without Constraint')
disp(num2str(CI_unc));

