function quadvals = MakeQuadPoints_with_jumps(knots,nquad,changepoints,delta)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% functin MakeQuadPoints
%
% Creates quadrature points and values for using Simpson's rule in a
% Bspline basis.   When delta and changepoints are omitted this function is
% the same as MakeQuadPoints.m
%
% INPUTS:
%
% knots  - the knots of the basis
%
% nquad  - number of quadrature points to use
%
% changepoints - positions where there is a break in the function to integrate.
%                Omit a neighbourhood of size delta before and after these
%                changepoints.
% delta       - half the neighbourhood size.  Default value is 1e-6 *
%               smallest distance between unique knots.
% OUTPUT:
% 
% quadvals - a 2-dimensional array containing the quadrature points to use
%            in the first dimension and the corresponding quadrature
%            weights in the second.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

breaks  = unique(knots);
nbreaks = length(breaks);

if nargin == 3   % delta not given but needed
    delta = 1e-2 * min(diff(breaks));
elseif nargin >3 && isempty(delta)
    delta = 1e-2 * min(diff(breaks));
end
if nargin<3   % There are no changepoints
    changepoints = [];
end


if nargin>2  % there are changepoints to avoid.
    breaks  = unique([breaks,changepoints-delta,changepoints+delta]);
    nbreaks = length(breaks);
end

quadpts1 = linspace(breaks(1),breaks(2),nquad)';
quadwts1 = ones(nquad,1);
quadwts1(2:2:nquad-1) = 4;
quadwts1(3:2:nquad-2) = 2;
quadwts1 = ((breaks(2)-breaks(1))/(nquad-1)).*quadwts1/3;
quadvals = [quadpts1, quadwts1];

for i=3:nbreaks
    quadptsi = linspace(breaks(i-1),breaks(i),nquad)';
    quadwtsi = ones(nquad,1);    
    quadwtsi(3:2:nquad-2) = 2;
    quadwtsi(2:2:nquad-1) = 4;
    quadwtsi = ((breaks(i)-breaks(i-1))/(nquad-1)).*quadwtsi/3;
    quadvals = [quadvals;[quadptsi, quadwtsi]];
end
if ~isempty(changepoints)
    for j=1:length(changepoints)
           quadvals((quadvals(:,1)>changepoints(j)-delta &... 
                         quadvals(:,1)<changepoints(j)+delta),2)=0;
    end
    % Since we create a new 'knot' at the edges of the neighbouhood, we
    % have 2 quadvalues at each of the edges.  But we want to eliminate
    % only one of them.
    quadvals(find(quadvals(:,1)==changepoints(j)-delta,1,'last'),2)  = 0;
    quadvals(find(quadvals(:,1)==changepoints(j)+delta,1,'last'),2) = 0;
end
end