demo_optsup <- function() {
    ## The opportunities suppression experiment aims to revise the way Adwords
    ## show opportunities to the advertisers. Prior to this experiment, there
    ## was another effort on re-ranking the opportunities, but the experiment
    ## didn't favor the new ranking algorithm. This suppression experiment
    ## can be viewed as an extreme version of re-ranking: in the treatment
    ## group, some of the opportunities are actually dropped rather than
    ## sinked to bottom. The response quantity of interest is the
    ## click-through-rate (CTR) of opportunites.

    ## Although the customers are randomized in this experiment, a systematic
    ## software bug allows some treated customers exposed to both
    ## non-suppressed and suppressed opportunities. These customers did not
    ## occur at random, because the bug is related to whether the customer
    ## often gets multiple suggestions for a single campaign. The goal of
    ## this study is to ignore this sub-population and rebalance the
    ## treatment and control groups to compare their CTR. Note that due to
    ## the bug the experiment lasted for only one day, so there are fewer data
    ## points in this data set.

    ## First read the data.
    opt.data <- read.csv(
        gfile("/cns/ie-d/home/dancsi/opt_suppression_more_features-20140807.csv"),
        header = TRUE)

    ## Visualization of the bug
    table(opt.data$experiment_name)
    plot(jitter(view_type_a_sum) ~ jitter(view_type_s_sum),
         data = opt.data[opt.data$experiment_name == "R_SUPPA", ],
         col = 25 + 9 * (view_type_a_sum * view_type_s_sum > 0),
         main = "Scatterplot of suppressed customers' views")
    legend("topright",
           legend = c("affected by the bug",
               "not affected by the bug"),
           col = c("red", "black"), pch = 1)

    ## Figure \ref{fig:opt-visualize} visualize the bug mentioned above. Each
    ## point in the scatterplot represents one customer in the treatment group and the x and y axes
    ## are the total number of type S and type A opportunities the customer
    ## saw during the experiment. If the experiment was conducted correctly,
    ## no customer in the treatment group should see at least one type A
    ## and at least type S opportunities. Note that it is okay if the treated
    ## customer has only type S views, because we don't suppress anything if
    ## no type A opportunity is avaiable.

    ## The methods we will use are
    ## method.list = c("unif,pop,none,none",
    ##     "glm,pop,none,none", "ebal,nr,none,none", "cbps,pop,none,none",
    ##     "glm,pop,wls,none", "ebal,nr,wls,none", "cbps,pop,wls,none",
    ##     "glm,pop,lm,dr", "cbps,pop,lm,dr")

    ## Now let's take out the bug-affected customers
    data <- subset(data.table(opt.data), experiment_name == "R_CTRLA" |
                       view_type_a_sum > 0 | view_type_s_sum == 0)
    data <- data.frame(data)
    ## and transform it into the format \texttt{covalign} recognizes. Notice
    ## that there
    ## are two possible definition of click through rate: mean(individual
    ## CTR) or mean(click)/mean(view).

    ## \subsubsection{Mean of individual CTR}

    ## This definition is easier for \texttt{covalign} to estimate.
    data.ratio <- list(T = as.numeric(data$experiment_name) - 1,
                       features = log(data.matrix(data[,
                           grepl("20140618", names(data))]) + 1, 2),
                       Y = data$click_type_a_sum / data$view_type_a_sum)
    data.ratio$Y[is.nan(data.ratio$Y)] <- 0
    data.ratio$stratum <- cut(data.ratio$features[, "spend_7_day_usd_20140618"],
                              c(seq(0, 12, by=2), Inf),
                              right = FALSE)
    table(data.ratio$stratum)
    opt.result <- get.effect.obdata(data.ratio,
                                    effect = "ATE",
                                    stratified = TRUE,
                                    method.list = method.list,
                                    print.level = 0)
    options(digits = 3)
    print(opt.result$effect.est)

    ## There are two ways to get the standard error. For doubly robust
    ## estimators of ATE, there is a nice sandwich variance estimator (See
    ## \citep{Lunceford2004}). I implemented this in the function \texttt{get.effect.se}.
    effect.est.se.formula <- rep(0, length(levels(data.ratio$stratum)))
    for (i in 1:length(levels(data.ratio$stratum))) {
        diagnostics.object <- opt.result$output[[i]][["glm,pop,lm,dr"]]
        effect.est.se.formula[i] <- get.effect.se(diagnostics.object)
    }

    ## The other way to get se is by bootstrap
    B <- 50
    effect.est.bs <- array(0, c(B, length(levels(data.ratio$stratum)),
                                length(method.list)))
    n <- length(data.ratio$stratum)
    for (b in 1:B) {
        ## print(paste("Bootstrap iteration:", b))
        s <- sample(1:n, n, replace = TRUE)
        effect.est.bs[b, , ] <-
            get.effect.obdata(list(T = data.ratio$T[s],
                                   Y = data.ratio$Y[s],
                                   features = data.ratio$features[s, ],
                                   stratum = data.ratio$stratum[s]),
                              effect = "ATE",
                              stratified = TRUE,
                              method.list = method.list,
                              print.level = 0)$effect.est
    }
    effect.est.se <- apply(effect.est.bs, c(2, 3), sd, na.rm = TRUE)
    effect.est <- opt.result$effect.est
    rownames(effect.est.se) <- rownames(effect.est)
    colnames(effect.est.se) <- colnames(effect.est)

    ## Now we can compare the two methods
    cbind(formula = effect.est.se.formula,
          bootstrap = effect.est.se[, "glm,pop,lm,dr"])
    ## The standard errors are in the same scale, but the bootstrap estimate
    ## is slightly larger than M-estimation formula-based estimate.

    ## Plot the estimated causal effect
    methods <- c("unif,pop,none,none", "glm,pop,none,none", "ebal,nr,none,none",
                 "cbps,pop,none,none", "glm,pop,lm,dr", "ebal,nr,wls,none",
                 "glm,pop,wls,none")
    df.est <- melt(effect.est[, methods])
    df.se <- melt(effect.est.se[, methods])
    df <- data.frame(stratum = df.est$X1, method = df.est$X2,
                     est = df.est$value, se = df.se$value)
    df$stratum <- factor(df$stratum, levels(data.ratio$stratum))
    p <- ggplot(df) + aes(x = stratum, y = est) +
        geom_point(aes(group = method, color = method), shape = 21, size = 3) +
            geom_errorbar(aes(ymax = est + 2 * se,
                              ymin = est - 2 * se,
                              group = method, color = method))
    p <- p + theme(text = element_text(size = 15)) +
        ggtitle("Opportunity Suppression Experiment After Adjustment") +
            ylab("difference in CTR") + xlab("Stratum by log2(previous week spend)")
    p

    ## In this plot, WLS is not very stable (especially with entropy
    ## balancing weights) and all the other methods are pretty much the
    ## same. The CTR increase is about 20\% to 30\% and is significant across
    ## all spend strata.
    methods <- c("unif,pop,none,none", "glm,pop,lm,dr")
    df.est <- melt(effect.est[, methods])
    df.se <- melt(effect.est.se[, methods])
    df <- data.frame(stratum = df.est$X1, method = df.est$X2,
                     est = df.est$value, se = df.se$value)
    levels(df$method)[levels(df$method) == "unif,pop,none,none"] <- "before adjustment"
    levels(df$method)[levels(df$method) == "glm,pop,lm,dr"] <- "after adjustment"
    df$stratum <- factor(df$stratum, levels(data.ratio$stratum))
    p <- ggplot(df) + aes(x = stratum, y = est) +
        geom_point(aes(group = method, color = method), shape = 21, size = 3) +
            geom_errorbar(aes(ymax = est + 2 * se,
                              ymin = est - 2 * se,
                              group = method, color = method))
    p <- p + theme(text = element_text(size = 15)) +
        ggtitle("Opportunity Suppression Experiment After Adjustment") +
            ylab("difference in CTR") + xlab("Stratum by log2(previous week spend)") +
                ylim(-0.3,0.8)
    p

    ## For more regression diagnostics, please see the demo with hedwig data.

    ## \subsection{mean(click)/mean(view)}
    ## The second definition of CTR is avg(individual click) / avg(individual view).
    data.click <- list(T = as.numeric(data$experiment_name) - 1,
                       features = log(data.matrix(data[,
                           grepl("20140618", names(data))]) + 1, 2),
                       Y = data$click_type_a_sum)
    data.view <- list(T = as.numeric(data$experiment_name) - 1,
                      features = log(data.matrix(data[,
                          grepl("20140618", names(data))]) + 1, 2),
                      Y = data$view_type_a_sum)
    data.click$stratum <- cut(data.click$features[, "spend_7_day_usd_20140618"],
                              c(seq(0, 12, by=2), Inf),
                              right = FALSE)
    data.view$stratum <- data.click$stratum
    opt.click.result <- get.effect.obdata(data.click,
                                          effect = "ATE",
                                          stratified = TRUE,
                                          method.list = method.list,
                                          print.level = 0)
    opt.view.result <- get.effect.obdata(data.view,
                                         effect = "ATE",
                                         stratified = TRUE,
                                         method.list = method.list,
                                         print.level = 0)

    ## One may get the difference in click and view by
    options(digits = 3)
    print(opt.click.result$effect.est)
    print(opt.view.result$effect.est)

    ## The difference of CTR in this definition is
    options(digits = 3)
    print(opt.click.result$mean1.est / opt.view.result$mean1.est -
              opt.click.result$mean0.est / opt.view.result$mean0.est)

    ## Now we can use bootstrap to get the standard error like before. I will
    ## omit this here.

}
